#ifndef _fma_probe_h_
#define _fma_probe_h_

#include "lf_myri_packet.h"

/*
 * Definitions for sending and tracking probes
 */
enum {		/* probe types */
  FMA_PT_NIC_SCOUT,
  FMA_PT_TAGGED_XBAR_VERIFY,
  FMA_PT_TAGGED_XBAR_SCOUT,
  FMA_PT_XBAR_VERIFY,
  FMA_PT_XBAR_SCOUT,
  FMA_PT_NIC_VERIFY,
  FMA_PT_LINK_VERIFY,
  FMA_PT_EMPTY_PORT_CHECK_XBAR,
  FMA_PT_EMPTY_PORT_CHECK_NIC,
  FMA_PT_FMA_QUERY,
  FMA_PT_ANY		/* matches any, a wildcard */
};

#define FMA_FMA_QUERY_TIMEOUT 250
#define FMA_FMA_QUERY_RETRY 3

#define FMA_XBAR_VERIFY_TIMEOUT 100
#define FMA_XBAR_VERIFY_RETRY 20

#define FMA_XBAR_SCOUT_TIMEOUT 30
#define FMA_XBAR_SCOUT_RETRY 3

#define FMA_PROBE_HASH_SIZE 1024
#define FMA_PROBE_HASH(S) ((S) & (FMA_PROBE_HASH_SIZE-1))

struct fma_probe_desc;		/* forward */
typedef void (*fma_probe_resp_rtn_t)(struct fma_probe_desc *pdp,
                                      void *pkt, int len);
typedef void (*fma_probe_fail_rtn_t)(struct fma_probe_desc *);

struct fma_probe_desc {
  uint32_t pd_serial;		/* serial number of expected packet */
  int pd_type;			/* type of probe */

  /* retry information */
  int retries_left;			/* number of times to retry */
  int pd_timeout;			/* timeout interval */
  struct fma_sync_timer *pd_sync_timer;	/* timeout event */
  struct lf_event *pd_timer;		/* general use timer */
  fma_probe_resp_rtn_t resp_rtn;	/* call if got response */
  fma_probe_fail_rtn_t fail_rtn;	/* call if all retries fail */
  int persistant;			/* if false, free when fail or resp */

  /* NIC and port from which probe was sent */
  struct fma_nic_info *origin;
  int pd_port;

  union {
    struct myri_nic_scout nic_scout;	/* NIC scout sent */
    struct fma_myri_packet fma_pkt;	/* fma packet sent */
    struct fma_xbar_scout_pkt xbar_scout_pkt;	/* xbar scout packet */
  } packet;
  int pkt_len;

  unsigned char pd_route[MYRI_MAX_ROUTE_LEN];
  int pd_route_len;

  union {

    /* FMA_PT_TAGGED_XBAR_VERIFY - verify a tagged xbar is where / what
     * it is supposed to be */
    struct fma_tagged_xbar_verify {
      struct lf_xbar *xbar;		/* xbar we expect to see */
      int port;				/* port we expect to see */
    } tagged_xbar_verify;

    /* FMA_PT_NIC_VERIFY - verify a NIC is where it should be */
    struct fma_nic_verify {
      struct lf_nic *nv_nic;		/* nic we expect to see */
      int nv_port;			/* port we expect to see */
    } nic_verify;

    /* FMA_PT_LINK_VERIFY - verify a link by sending a msg to a NIC using it */
    struct fma_link_verify {
      struct lf_nic *lv_nic;		/* nic we expect to see */
      int lv_nic_port;			/* port we expect to see */

      struct lf_xbar *lv_xbar;		/* xbar and port under test */
      int lv_xbar_port;
      int lv_link_down;			/* If true, link is down */
    } link_verify;

    /* FMA_PT_XBAR_VERIFY - verify an xbar is where it should be */

    /* FMA_PT_EMPTY_PORT_VERIFY - check "empty" ports */
    struct fma_empty_port_verify {
      union lf_node *np;
      int port;
    } empty_port_verify;

    /* FMA_PT_FMA_QUERY - query for FMA info */
    struct fma_query_probe {
      struct lf_nic *nicp;		/* NIC for this query */
    } query;

  } u;

  /* doubly linked list for easy insert/remove */
  struct fma_probe_desc *user_prev;
  struct fma_probe_desc *user_next;
  struct fma_probe_desc *pd_prev;
  struct fma_probe_desc *pd_next;
};

/*
 * Some inlines for list manipulation
 */
static inline void
fma_link_probe_descriptor(
  struct fma_probe_desc *pdp)
{
  struct fma_probe_desc *anchor;
  int index;

  index = FMA_PROBE_HASH(pdp->pd_serial);

  anchor = &A.myri->probe_anchors[index];

  /* link this descriptor to end of list */
  pdp->pd_next = anchor;
  pdp->pd_prev = anchor->pd_prev;

  /* insert into the list */
  pdp->pd_prev->pd_next = pdp;
  pdp->pd_next->pd_prev = pdp;
}

static inline void
fma_unlink_probe_descriptor(
  struct fma_probe_desc *pdp)
{
  /* remove from the list, checking for NULL to prevent double unlinks */
  if (pdp->pd_next != NULL) {
    pdp->pd_prev->pd_next = pdp->pd_next;
    pdp->pd_next->pd_prev = pdp->pd_prev;
    pdp->pd_next = NULL;
  }
}

/*
 * prototypes
 */
void fma_got_nic_scout(struct fma_nic_info *nip, int port,
                       struct myri_nic_scout *pkt, int len);
void fma_got_xbar_scout_pkt(struct fma_nic_info *nip, int port,
                       struct fma_xbar_scout_pkt *pkt, int len);
void fma_got_nic_scout_resp(struct fma_nic_info *nip, int port,
                             struct myri_nic_scout_reply *pkt, int len);
void fma_got_nic_scout_resp(struct fma_nic_info *nip, int port,
  struct myri_nic_scout_reply *pkt, int len);
void fma_got_probe_resp(struct fma_nic_info *nip, int port, int serial,
	    lf_mac_addr_t *sender_mac, void *pkt, int len);

struct fma_probe_desc *fma_create_tagged_xbar_scout_probe(
  struct fma_nic_info *nip,
  int port, int type, fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn, unsigned char *route, int route_len);
struct fma_probe_desc *fma_create_xbar_scout_probe(struct fma_nic_info *nip,
  int port, int type, fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn, unsigned char *route, int route_len);
struct fma_probe_desc *fma_create_nic_scout_probe(struct fma_nic_info *nip,
  int port, int type, fma_probe_resp_rtn_t resp_rtn,
  fma_probe_fail_rtn_t fail_rtn, unsigned char *route, int route_len);
void fma_send_probe_packet(struct fma_probe_desc *pdp);
void fma_probe_timeout(void *vpdp);
void fma_set_probe_levels(int level);
void fma_fill_xbar_scout(struct fma_xbar_scout_pkt *pkt,
		        struct fma_nic_info *nip, int port, int serial);
void fma_fill_tagged_xbar_scout(struct fma_xbar_scout_pkt *pkt,
		       struct fma_nic_info *nip, int port, int serial);
void fma_fill_nic_generic_scout(struct myri_nic_scout *pkt,
				unsigned char *route, int route_len);
void fma_fill_nic_verify_scout(struct myri_nic_scout *pkt,
		               struct fma_nic_info *nip, int port, int serial,
			       unsigned char *route, int route_len);

#endif /* _fma_probe_h_ */
